<?php
/**
 * SuperSite2 / ResellerClub - Custom Payment Gateway
 * CinetPay Checkout v2 integration (XAF only)
 *
 * IMPORTANT:
 * 1) Set RC_SECURE_KEY (ResellerClub secure key).
 * 2) Set CINETPAY_SITE_ID / CINETPAY_API_KEY / CINETPAY_SECRET_KEY.
 * 3) Configure the gateway in ResellerClub with Checksum Algorithm = MD5.
 *
 * CinetPay notes:
 * - notification url must accept GET/POST and return HTTP 200 OK (no redirects).
 * - return_url must accept GET/POST and return HTTP 200 OK.
 *
 * Docs:
 * - https://docs.cinetpay.com/api/1.0-en/checkout/initialisation
 * - https://docs.cinetpay.com/api/1.0-en/checkout/notification
 * - https://docs.cinetpay.com/api/1.0-en/checkout/retour
 */

declare(strict_types=1);

/** ============ ResellerClub ============ */
const RC_SECURE_KEY = 'REPLACE_WITH_RESELLERCLUB_SECURE_KEY';

/** ============ CinetPay ============ */
const CINETPAY_SITE_ID    = 'REPLACE_WITH_SITE_ID';
const CINETPAY_API_KEY    = 'REPLACE_WITH_API_KEY';
const CINETPAY_SECRET_KEY = 'REPLACE_WITH_SECRET_KEY'; // Not required for /v2/payment init/check, kept for your records.
const CINETPAY_LANG       = 'fr';
const CINETPAY_CHANNELS   = 'ALL';

/**
 * Force output currency to XAF (as requested).
 * WARNING: If your SuperSite2 shows another currency, ensure the amount you receive is already the XAF value.
 */
const CINETPAY_OUT_CURRENCY = 'XAF';

/** Logging */
const RC_LOG_FILE = __DIR__ . '/logs/cinetpay.log';
const RC_STATE_DIR = __DIR__ . '/_tx_state';

function rc_log(string $msg, array $ctx = []): void {
    $line = '[' . date('Y-m-d H:i:s') . '] ' . $msg;
    if (!empty($ctx)) $line .= ' | ' . json_encode($ctx, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    @file_put_contents(RC_LOG_FILE, $line . PHP_EOL, FILE_APPEND);
}

function rc_param(string $key, ?string $default = null): ?string {
    if (isset($_POST[$key]) && is_string($_POST[$key])) return trim($_POST[$key]);
    if (isset($_GET[$key]) && is_string($_GET[$key]))  return trim($_GET[$key]);
    return $default;
}

/** Robust number parsing for amounts (supports "1 234,56", "1234.56", etc.) */
function rc_to_number(string $s): float {
    $s = trim($s);
    $s = str_replace(["\xC2\xA0", ' '], '', $s); // remove non-breaking space and space
    $s = preg_replace('/[^0-9\.,\-]/', '', $s);

    $hasComma = strpos($s, ',') !== false;
    $hasDot   = strpos($s, '.') !== false;

    if ($hasComma && $hasDot) {
        // assume commas are thousands separators: "1,234.56"
        $s = str_replace(',', '', $s);
    } elseif ($hasComma && !$hasDot) {
        // comma as decimal separator: "1234,56"
        $s = str_replace(',', '.', $s);
    }

    if ($s === '' || $s === '-' || $s === '.') return 0.0;
    return (float)$s;
}

/**
 * Amount rule requested:
 * - remove decimals entirely
 * - round UP to the nearest multiple of 5 (5997 -> 6000, 8982 -> 8985)
 */
function cinetpay_amount_xaf(string $raw): int {
    $n = rc_to_number($raw);
    $int = (int)floor($n); // drop decimals
    if ($int <= 0) throw new RuntimeException("Montant invalide (<=0) après conversion: {$raw}");
    $mod = $int % 5;
    if ($mod !== 0) $int += (5 - $mod); // ceil to multiple of 5
    return $int;
}

function rc_base_url(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $script = $_SERVER['SCRIPT_NAME'] ?? '/';
    $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');
    return $scheme . '://' . $host . ($dir ? $dir . '/' : '/');
}

/** State fallback */
function rc_ensure_state_dir(): void {
    if (!is_dir(RC_STATE_DIR)) @mkdir(RC_STATE_DIR, 0755, true);
}
function rc_save_state(string $transId, array $state): void {
    rc_ensure_state_dir();
    $safe = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', $transId);
    @file_put_contents(RC_STATE_DIR . "/tx_{$safe}.json", json_encode($state, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
}
function rc_load_state(string $transId): ?array {
    rc_ensure_state_dir();
    $safe = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', $transId);
    $path = RC_STATE_DIR . "/tx_{$safe}.json";
    if (!is_file($path)) return null;
    $json = @file_get_contents($path);
    $data = json_decode((string)$json, true);
    return is_array($data) ? $data : null;
}

/**
 * verifyChecksum has 2 known variants depending on RC kit/version.
 * We attempt both.
 */
function verifyChecksumFlexible(array $p, string $key, string $checksum): bool {
    $c1 = md5(
        ($p['paymenttypeid'] ?? '') . '|' . ($p['transid'] ?? '') . '|' . ($p['userid'] ?? '') . '|' . ($p['usertype'] ?? '') . '|' . ($p['transactiontype'] ?? '') . '|' .
        ($p['invoiceids'] ?? '') . '|' . ($p['debitnoteids'] ?? '') . '|' . ($p['description'] ?? '') . '|' .
        ($p['sellingcurrencyamount'] ?? '') . '|' . ($p['accountingcurrencyamount'] ?? '') . '|' . $key
    );

    $c2 = md5(
        ($p['paymenttypeid'] ?? '') . '|' . ($p['transid'] ?? '') . '|' . ($p['userid'] ?? '') . '|' . ($p['usertype'] ?? '') . '|' . ($p['transactiontype'] ?? '') . '|' .
        ($p['invoiceids'] ?? '') . '|' . ($p['debitnoteids'] ?? '') . '|' . ($p['description'] ?? '') . '|' .
        ($p['sellingcurrency'] ?? '') . '|' . ($p['sellingcurrencyamount'] ?? '') . '|' .
        ($p['accountingcurrency'] ?? '') . '|' . ($p['accountingcurrencyamount'] ?? '') . '|' . $key
    );

    $in = strtolower(trim($checksum));
    return ($in === strtolower($c1)) || ($in === strtolower($c2));
}

function generateChecksum(string $transId, string $sellingAmount, string $accountingAmount, string $status, string $rkey, string $key): string {
    return md5($transId . '|' . $sellingAmount . '|' . $accountingAmount . '|' . $status . '|' . $rkey . '|' . $key);
}

/** ---- CinetPay ---- */
function http_post_json(string $url, array $payload, int $timeoutSeconds = 30): array {
    $ch = curl_init($url);
    if ($ch === false) throw new RuntimeException("cURL init failed");

    $json = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
        CURLOPT_POSTFIELDS     => $json,

        // Network hardening (fixes many shared-hosting issues):
        // - Force IPv4 (some hosts have broken IPv6 routing/DNS)
        // - Increase connect timeout
        // - Increase overall timeout
        CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        CURLOPT_CONNECTTIMEOUT => 30,
        CURLOPT_TIMEOUT        => max($timeoutSeconds, 60),

        // Optional: some old stacks need explicit TLS 1.2
        CURLOPT_SSLVERSION     => CURL_SSLVERSION_TLSv1_2,
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $http = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) throw new RuntimeException("cURL error: $err");

    $decoded = json_decode((string)$resp, true);
    if (!is_array($decoded)) throw new RuntimeException("Non-JSON response (HTTP $http): " . substr((string)$resp, 0, 300));

    return ['http' => $http, 'data' => $decoded];
}

function cinetpay_init(string $transactionId, int $amountXaf, string $description, string $notifyUrl, string $returnUrl, array $customer = [], string $metadata = ''): string {
    $payload = [
        'apikey'         => CINETPAY_API_KEY,
        'site_id'        => CINETPAY_SITE_ID,
        'transaction_id' => $transactionId,
        'amount'         => $amountXaf,
        'currency'       => CINETPAY_OUT_CURRENCY,
        'description'    => $description,
        'notify_url'     => $notifyUrl,
        'return_url'     => $returnUrl,
        'channels'       => CINETPAY_CHANNELS,
        'lang'           => CINETPAY_LANG,
    ];
    if ($metadata !== '') $payload['metadata'] = $metadata;

    foreach ($customer as $k => $v) {
        if ($v !== null && $v !== '') $payload[$k] = (string)$v;
    }

    rc_log("CINETPAY_INIT", ['payload' => $payload]);

    $res = http_post_json('https://api-checkout.cinetpay.com/v2/payment', $payload);
    rc_log("CINETPAY_INIT_RES", ['http' => $res['http'], 'data' => $res['data']]);

    $url = $res['data']['data']['payment_url'] ?? null;
    if (!is_string($url) || $url === '') {
        $msg = $res['data']['message'] ?? 'unknown';
        $desc = $res['data']['description'] ?? '';
        throw new RuntimeException("CinetPay init failed: $msg $desc");
    }
    return $url;
}

function cinetpay_check(string $transactionId): array {
    $payload = [
        'transaction_id' => $transactionId,
        'site_id'        => CINETPAY_SITE_ID,
        'apikey'         => CINETPAY_API_KEY,
    ];
    rc_log("CINETPAY_CHECK", ['payload' => $payload]);

    $res = http_post_json('https://api-checkout.cinetpay.com/v2/payment/check', $payload);
    rc_log("CINETPAY_CHECK_RES", ['http' => $res['http'], 'data' => $res['data']]);
    return $res['data'];
}

function cinetpay_to_rc_status(?string $status, ?string $message = null): string {
    $s = strtoupper((string)$status);
    $m = strtoupper((string)$message);

    if ($s === 'ACCEPTED') return 'Y';
    if ($s === 'REFUSED') return 'N';
    if (str_contains($m, 'CANCEL')) return 'N';
    return 'P';
}


/**
 * Quick connectivity debug endpoint (optional).
 * Create a file debug.php with:
 *   <?php require 'functions.php'; rc_net_debug();
 */
function rc_net_debug(): void {
    header('Content-Type: text/plain; charset=utf-8');
    echo "PHP: " . PHP_VERSION . "\n";
    echo "cURL: " . (function_exists('curl_version') ? json_encode(curl_version(), JSON_UNESCAPED_SLASHES) : 'missing') . "\n";
    echo "DNS api-checkout.cinetpay.com: ";
    $ip = gethostbyname('api-checkout.cinetpay.com');
    echo $ip . "\n";
    echo "Try HTTPS HEAD...\n";
    $ch = curl_init('https://api-checkout.cinetpay.com/');
    curl_setopt_array($ch, [
        CURLOPT_NOBODY => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
        CURLOPT_CONNECTTIMEOUT => 15,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
    ]);
    $ok = curl_exec($ch);
    $err = curl_error($ch);
    $http = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    echo "HTTP: {$http}\n";
    if ($ok === false) echo "ERR: {$err}\n";
}
