<?php
declare(strict_types=1);

/**
 * ResellerClub / Supersite2 Payment Gateway helpers + Notch Pay integration
 *
 * IMPORTANT:
 * - Replace RC_SECURE_KEY with your Supersite2 "Secure Key"
 * - Replace NOTCHPAY_API_KEY with your Notch Pay API key
 * - (Optional) Replace NOTCHPAY_WEBHOOK_HASH if you use webhooks verification
 */

const DEBUG = false;

// === Supersite2 / ResellerClub ===
const RC_SECURE_KEY = 'REPLACE_WITH_YOUR_SUPERSITE2_SECURE_KEY';

// === Notch Pay ===
// Docs: https://developer.notchpay.co/api-reference/payments  
const NOTCHPAY_API_KEY = 'REPLACE_WITH_YOUR_NOTCHPAY_API_KEY';
const NOTCHPAY_BASE_URL = 'https://api.notchpay.co';

// If you configure a webhook in NotchPay dashboard, you can verify it using this hash
// Docs: webhook verification uses HMAC SHA-256 + `x-notch-signature` header 
const NOTCHPAY_WEBHOOK_HASH = 'REPLACE_WITH_YOUR_NOTCHPAY_WEBHOOK_HASH';

// Default values if Supersite2 does not provide currency/channel (Supersite2 sample doesn’t send currency code).
const NOTCHPAY_DEFAULT_CURRENCY = 'XAF'; // change if needed

// Optional "locked_*" parameters (leave empty string to not send)
const NOTCHPAY_LOCKED_COUNTRY = 'CM';   // ex: 'CM'
const NOTCHPAY_LOCKED_CHANNEL = '';     // ex: 'cm.mtn' or 'cm.orange'
const NOTCHPAY_LOCKED_CURRENCY = '';    // ex: 'XAF'

// HTTP
const NOTCHPAY_TIMEOUT = 30;

// Storage (file-based) to survive session loss during callback/webhook
const STORAGE_DIR = __DIR__ . '/storage';

/**
 * HTML escape helper
 */
function h(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

/**
 * Compute a safe base URL to this module directory
 */
function module_base_url(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $dir = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '/')), '/');
    return $scheme . '://' . $host . $dir;
}

/**
 * Convert an amount to the smallest currency unit.
 * Notch Pay expects smallest unit for the currency 
 *
 * - XAF/XOF/… (0 decimals): keep as integer
 * - Most other currencies: multiply by 100
 */
function amount_to_smallest_unit($amount, string $currency): int {
    $amount = (string)$amount;
    $amount = str_replace(',', '.', $amount);
    $value = (float)$amount;

    $zeroDecimal = [
        'XAF','XOF','GNF','RWF','BIF','DJF','JPY','KRW','VND'
    ];
    if (in_array(strtoupper($currency), $zeroDecimal, true)) {
        return (int)round($value);
    }
    return (int)round($value * 100);
}

/**
 * Supersite2: Verify checksum received from foundation.
 */
function verifyChecksum(
    string $paymentTypeId,
    string $transId,
    string $userId,
    string $userType,
    string $transactionType,
    string $invoiceIds,
    string $debitNoteIds,
    string $description,
    string $sellingCurrencyAmount,
    string $accountingCurrencyAmount,
    string $key,
    string $checksum
): bool {
    $checksumString =
        $paymentTypeId . '|' .
        $transId . '|' .
        $userId . '|' .
        $userType . '|' .
        $transactionType . '|' .
        $invoiceIds . '|' .
        $debitNoteIds . '|' .
        $description . '|' .
        $sellingCurrencyAmount . '|' .
        $accountingCurrencyAmount . '|' .
        $key;

    $generated = md5($checksumString);
    return strtolower($generated) === strtolower($checksum);
}

/**
 * Supersite2: Generate checksum to send back to foundation after payment completion.
 */
function generateChecksum(
    string $transId,
    string $sellingCurrencyAmount,
    string $accountingCurrencyAmount,
    string $status,
    string $rkey,
    string $key
): string {
    $checksumString =
        $transId . '|' .
        $sellingCurrencyAmount . '|' .
        $accountingCurrencyAmount . '|' .
        $status . '|' .
        $rkey . '|' .
        $key;

    return md5($checksumString);
}

/**
 * =============== Storage helpers (file-based) ===============
 */
function storage_ensure(): void {
    if (!is_dir(STORAGE_DIR)) {
        @mkdir(STORAGE_DIR, 0775, true);
    }
}

function tx_filename(string $transId): string {
    storage_ensure();
    $safe = preg_replace('/[^A-Za-z0-9_.-]/', '_', $transId);
    return rtrim(STORAGE_DIR, '/\\') . '/tx_' . $safe . '.json';
}

function tx_store(string $transId, array $data): void {
    $file = tx_filename($transId);
    $data['_stored_at'] = time();
    $json = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if ($json === false) {
        return;
    }
    $tmp = $file . '.tmp';
    file_put_contents($tmp, $json, LOCK_EX);
    @rename($tmp, $file);
}

function tx_load(string $transId): ?array {
    $file = tx_filename($transId);
    if (!is_file($file)) return null;
    $raw = file_get_contents($file);
    if ($raw === false) return null;
    $data = json_decode($raw, true);
    return is_array($data) ? $data : null;
}

function tx_patch(string $transId, array $patch): void {
    $data = tx_load($transId) ?? [];
    foreach ($patch as $k => $v) $data[$k] = $v;
    tx_store($transId, $data);
}

/**
 * =============== Notch Pay helpers ===============
 */
function notchpay_request(string $method, string $path, ?array $payload = null): array {
    $url = rtrim(NOTCHPAY_BASE_URL, '/') . '/' . ltrim($path, '/');
    $ch = curl_init($url);

    $headers = [
        'Authorization: ' . NOTCHPAY_API_KEY,
        'Content-Type: application/json',
        'Accept: application/json',
    ];

    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_TIMEOUT => NOTCHPAY_TIMEOUT,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_CUSTOMREQUEST => strtoupper($method),
    ];

    if ($payload !== null) {
        $body = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        if ($body === false) $body = '{}';
        $opts[CURLOPT_POSTFIELDS] = $body;
    }

    curl_setopt_array($ch, $opts);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) {
        return ['ok' => false, 'http_code' => $code, 'error' => $err ?: 'cURL error', 'raw' => null];
    }

    $decoded = json_decode($resp, true);
    return [
        'ok' => ($code >= 200 && $code < 300),
        'http_code' => $code,
        'error' => null,
        'raw' => $resp,
        'json' => is_array($decoded) ? $decoded : null,
    ];
}

function notchpay_initialize_payment(array $payload): array {
    // POST /payments 
    return notchpay_request('POST', '/payments', $payload);
}

function notchpay_retrieve_payment(string $reference): array {
    // GET /payments/{reference} 
    return notchpay_request('GET', '/payments/' . rawurlencode($reference), null);
}

/**
 * Verify NotchPay webhook signature (HMAC SHA-256) 
 */
function notchpay_verify_webhook(string $rawPayload, string $signature): bool {
    if (NOTCHPAY_WEBHOOK_HASH === '' || NOTCHPAY_WEBHOOK_HASH === 'REPLACE_WITH_YOUR_NOTCHPAY_WEBHOOK_HASH') {
        // If not configured, skip verification.
        return true;
    }
    $expected = hash_hmac('sha256', $rawPayload, NOTCHPAY_WEBHOOK_HASH);
    // timing-safe compare
    return hash_equals($expected, $signature);
}
