<?php
declare(strict_types=1);

session_start();
require_once __DIR__ . '/functions.php';

/**
 * Supersite2 calls this file with params:
 * paymenttypeid, transid, userid, usertype, transactiontype, invoiceids, debitnoteids,
 * description, sellingcurrencyamount, accountingcurrencyamount, redirecturl, checksum
 *
 * We:
 * 1) verify checksum from foundation (ResellerClub)
 * 2) store all data (session + file storage)
 * 3) initialize a NotchPay payment and redirect user to authorization_url
 */

function get_str(array $src, string $key): string {
    $v = $src[$key] ?? '';
    if (is_array($v)) $v = implode(',', $v);
    $v = (string)$v;
    return trim($v);
}

$paymentTypeId            = urldecode(get_str($_GET, 'paymenttypeid'));
$transId                  = urldecode(get_str($_GET, 'transid'));
$userId                   = urldecode(get_str($_GET, 'userid'));
$userType                 = urldecode(get_str($_GET, 'usertype'));
$transactionType          = urldecode(get_str($_GET, 'transactiontype'));
$invoiceIds               = urldecode(get_str($_GET, 'invoiceids'));
$debitNoteIds             = urldecode(get_str($_GET, 'debitnoteids'));
$description              = urldecode(get_str($_GET, 'description'));
$sellingCurrencyAmount    = urldecode(get_str($_GET, 'sellingcurrencyamount'));
$accountingCurrencyAmount = urldecode(get_str($_GET, 'accountingcurrencyamount'));
$redirectUrl              = urldecode(get_str($_GET, 'redirecturl'));
$checksum                 = urldecode(get_str($_GET, 'checksum'));

// OPTIONAL (if you add them in your integration)
$currency                 = strtoupper(get_str($_GET, 'currency')) ?: NOTCHPAY_DEFAULT_CURRENCY;

// Secure key (ResellerClub / Supersite2)
$key = RC_SECURE_KEY;

if ($paymentTypeId === '' || $transId === '' || $redirectUrl === '' || $checksum === '') {
    http_response_code(400);
    echo "Bad request: missing required parameters.";
    exit;
}

if (!verifyChecksum(
    $paymentTypeId,
    $transId,
    $userId,
    $userType,
    $transactionType,
    $invoiceIds,
    $debitNoteIds,
    $description,
    $sellingCurrencyAmount,
    $accountingCurrencyAmount,
    $key,
    $checksum
)) {
    http_response_code(403);
    echo "Invalid checksum. Request rejected.";
    exit;
}

// Persist for callback
$tx = [
    'paymentTypeId' => $paymentTypeId,
    'transId' => $transId,
    'userId' => $userId,
    'userType' => $userType,
    'transactionType' => $transactionType,
    'invoiceIds' => $invoiceIds,
    'debitNoteIds' => $debitNoteIds,
    'description' => $description,
    'sellingCurrencyAmount' => $sellingCurrencyAmount,
    'accountingCurrencyAmount' => $accountingCurrencyAmount,
    'redirectUrl' => $redirectUrl,
    'currency' => $currency,
];

$_SESSION['rc_tx'] = $tx;
// For backward-compat with the original sample names
$_SESSION['redirecturl'] = $redirectUrl;
$_SESSION['transid'] = $transId;
$_SESSION['sellingcurrencyamount'] = $sellingCurrencyAmount;
$_SESSION['accountingcurencyamount'] = $accountingCurrencyAmount;

tx_store($transId, $tx);

// ---- Customer details (NotchPay create payment needs at least one customer identifier) 
$customerName  = trim(get_str($_POST, 'name')) ?: trim(get_str($_GET, 'name'));
$customerEmail = trim(get_str($_POST, 'email')) ?: trim(get_str($_GET, 'email'));
$customerPhone = trim(get_str($_POST, 'phone')) ?: trim(get_str($_GET, 'phone'));

$hasCustomer = ($customerEmail !== '' || $customerPhone !== '');

if ($_SERVER['REQUEST_METHOD'] !== 'POST' && !$hasCustomer) {
    // Minimal customer form, then we continue with NotchPay init
    $self = strtok($_SERVER['REQUEST_URI'], '?');
    $qs = http_build_query($_GET);
    $action = $self . ($qs ? ('?' . $qs) : '');
    ?>
    <!doctype html>
    <html lang="fr">
    <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <title>Paiement</title>
        <style>
            body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;background:#0b1220;color:#e6eefc;margin:0}
            .wrap{max-width:520px;margin:0 auto;padding:24px}
            .card{background:rgba(255,255,255,.06);border:1px solid rgba(255,255,255,.12);border-radius:14px;padding:18px}
            label{display:block;margin:10px 0 6px;font-size:14px;color:#c9d7ff}
            input{width:100%;padding:12px 12px;border-radius:10px;border:1px solid rgba(255,255,255,.18);background:rgba(0,0,0,.25);color:#fff}
            button{margin-top:14px;width:100%;padding:12px;border:0;border-radius:10px;background:linear-gradient(90deg,#00c2ff,#12d18e);color:#03131a;font-weight:700;cursor:pointer}
            .muted{opacity:.85;font-size:13px;line-height:1.4}
            .row{display:grid;grid-template-columns:1fr;gap:10px}
        </style>
    </head>
    <body>
    <div class="wrap">
        <div class="card">
            <h2 style="margin:0 0 10px">Informations de paiement</h2>
            <p class="muted" style="margin:0 0 14px">
                Pour continuer le paiement, renseignez un email ou un numéro de téléphone.
            </p>

            <div class="muted" style="margin-bottom:12px">
                <b>Référence:</b> <?=h($transId)?> <br>
                <b>Montant:</b> <?=h($sellingCurrencyAmount)?> <?=h($currency)?>
            </div>

            <form method="post" action="<?=h($action)?>">
                <label>Nom (optionnel)</label>
                <input name="name" placeholder="Votre nom">

                <label>Email</label>
                <input name="email" type="email" placeholder="[email protected]">

                <label>Téléphone</label>
                <input name="phone" placeholder="+2376XXXXXXXX">

                <button type="submit">Continuer vers le paiement</button>
            </form>
        </div>
    </div>
    </body>
    </html>
    <?php
    exit;
}

// Build NotchPay payment payload
$amountSmallest = amount_to_smallest_unit($sellingCurrencyAmount, $currency);

$callbackUrl = module_base_url() . '/postpayment.php?mode=callback';

// Use transId as reference, so the callback can verify it directly.
$payload = [
    'amount' => $amountSmallest,
    'currency' => $currency,
    'reference' => $transId,
    'description' => $description !== '' ? $description : ('Supersite2 payment ' . $transId),
    'callback' => $callbackUrl,
    'customer' => [
        'name' => $customerName !== '' ? $customerName : ('User ' . $userId),
        'email' => $customerEmail,
        'phone' => $customerPhone,
    ],
];

// Optional locks
if (NOTCHPAY_LOCKED_COUNTRY !== '') $payload['locked_country'] = NOTCHPAY_LOCKED_COUNTRY;
if (NOTCHPAY_LOCKED_CHANNEL !== '') $payload['locked_channel'] = NOTCHPAY_LOCKED_CHANNEL;
if (NOTCHPAY_LOCKED_CURRENCY !== '') $payload['locked_currency'] = NOTCHPAY_LOCKED_CURRENCY;

// Initialize payment
$resp = notchpay_initialize_payment($payload);

// Save request/response for troubleshooting
tx_patch($transId, [
    'notchpay_init_http' => $resp['http_code'] ?? null,
    'notchpay_init_raw'  => $resp['raw'] ?? null,
]);

if (!$resp['ok'] || empty($resp['json'])) {
    http_response_code(502);
    echo "NotchPay init failed. HTTP=" . (int)($resp['http_code'] ?? 0) . "<br>";
    if (DEBUG) echo "<pre>" . h((string)($resp['raw'] ?? $resp['error'] ?? '')) . "</pre>";
    exit;
}

$authorizationUrl = $resp['json']['authorization_url'] ?? '';
if ($authorizationUrl === '') {
    http_response_code(502);
    echo "NotchPay init failed: missing authorization_url.";
    if (DEBUG) echo "<pre>" . h(json_encode($resp['json'], JSON_PRETTY_PRINT)) . "</pre>";
    exit;
}

tx_patch($transId, [
    'notchpay_authorization_url' => $authorizationUrl,
]);

header('Location: ' . $authorizationUrl, true, 302);
exit;
