<?php
declare(strict_types=1);

session_start();
require_once __DIR__ . '/functions.php';

/**
 * This file handles:
 * - NotchPay callback redirect (mode=callback)
 * - NotchPay webhook notifications (mode=webhook) [optional]
 * - (legacy) manual "status" call (if you still want to test with ?status=Y/N)
 *
 * After we determine the final payment status, we redirect back to Supersite2
 * (foundation redirecturl) with the checksum parameters required by ResellerClub.
 */

$mode = $_GET['mode'] ?? '';

/**
 * -------- Webhook endpoint (optional) --------
 * Configure this URL in NotchPay dashboard:
 *   https://YOUR_DOMAIN/path/to/postpayment.php?mode=webhook
 *
 * Verify signature: HMAC SHA-256 with `x-notch-signature` header 
 */
if ($mode === 'webhook') {
    $raw = file_get_contents('php://input') ?: '';
    $sig = $_SERVER['HTTP_X_NOTCH_SIGNATURE'] ?? ($_SERVER['HTTP_X_NOTCHPAY_SIGNATURE'] ?? '');
    if ($sig === '' && function_exists('getallheaders')) {
        $hdrs = array_change_key_case(getallheaders(), CASE_LOWER);
        $sig = $hdrs['x-notch-signature'] ?? '';
    }

    if (!notchpay_verify_webhook($raw, $sig)) {
        http_response_code(403);
        echo "Invalid webhook signature";
        exit;
    }

    $json = json_decode($raw, true);
    // We don't rely on webhook payload structure (can evolve). We just persist for debug.
    if (is_array($json)) {
        $reference = (string)($json['reference'] ?? ($json['data']['reference'] ?? ($json['transaction']['reference'] ?? '')));
        if ($reference !== '') {
            tx_patch($reference, [
                'notchpay_webhook_received_at' => time(),
                'notchpay_webhook_payload' => $json,
            ]);
        }
    }

    http_response_code(200);
    echo "OK";
    exit;
}

// Try to detect reference (NotchPay redirects customer to callback?reference=...) 
$reference = $_GET['reference'] ?? ($_GET['transid'] ?? ($_SESSION['transid'] ?? ''));

if ($reference === '') {
    http_response_code(400);
    echo "Missing reference/transid.";
    exit;
}

$transId = (string)$reference;

// Load transaction context (prefer file storage, fallback to session)
$tx = tx_load($transId);
if (!$tx) $tx = $_SESSION['rc_tx'] ?? null;

$redirectUrl = (string)($tx['redirectUrl'] ?? ($_SESSION['redirecturl'] ?? ''));
$sellingCurrencyAmount = (string)($tx['sellingCurrencyAmount'] ?? ($_SESSION['sellingcurrencyamount'] ?? ''));
$accountingCurrencyAmount = (string)($tx['accountingCurrencyAmount'] ?? ($_SESSION['accountingcurencyamount'] ?? ''));

if ($redirectUrl === '' || $sellingCurrencyAmount === '' || $accountingCurrencyAmount === '') {
    http_response_code(500);
    echo "Transaction context not found (redirectUrl/amount).";
    echo "<br>Reference: " . h($transId);
    exit;
}

// Determine status:
// - NotchPay statuses: pending/processing/complete/failed/canceled/expired 
$status = '';
$notchStatus = '';
$verified = false;

if ($mode === 'callback') {
    // Verify payment status by calling GET /payments/{reference} 
    $resp = notchpay_retrieve_payment($transId);

    tx_patch($transId, [
        'notchpay_verify_http' => $resp['http_code'] ?? null,
        'notchpay_verify_raw'  => $resp['raw'] ?? null,
        'notchpay_callback_qs' => $_GET,
    ]);

    if ($resp['ok'] && !empty($resp['json']['transaction']['status'])) {
        $verified = true;
        $notchStatus = (string)$resp['json']['transaction']['status'];

        if ($notchStatus === 'complete') {
            $status = 'Y';
        } else {
            // Treat everything else as failed/not-paid for Supersite2 (Y/N model)
            $status = 'N';
        }
    } else {
        // If verify fails, fallback to "status" if provided (legacy testing)
        $status = strtoupper((string)($_REQUEST['status'] ?? 'N'));
        $status = ($status === 'Y') ? 'Y' : 'N';
    }
} else {
    // Legacy/manual testing: postpayment.php?status=Y or N
    $status = strtoupper((string)($_REQUEST['status'] ?? 'N'));
    $status = ($status === 'Y') ? 'Y' : 'N';
}

tx_patch($transId, [
    'final_status' => $status,
    'notchpay_status' => $notchStatus,
    'verified' => $verified,
]);

// Build Supersite2 redirect payload
$key = RC_SECURE_KEY;
$rkey = (string)random_int(100000, 999999999);

$checksum = generateChecksum(
    $transId,
    $sellingCurrencyAmount,
    $accountingCurrencyAmount,
    $status,
    $rkey,
    $key
);

// Auto-post back to foundation redirect URL
?>
<!doctype html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Retour paiement</title>
</head>
<body>
    <form name="f1" action="<?php echo h($redirectUrl); ?>" method="post">
        <input type="hidden" name="transid" value="<?php echo h($transId); ?>">
        <input type="hidden" name="status" value="<?php echo h($status); ?>">
        <input type="hidden" name="rkey" value="<?php echo h($rkey); ?>">
        <input type="hidden" name="checksum" value="<?php echo h($checksum); ?>">
        <input type="hidden" name="sellingamount" value="<?php echo h($sellingCurrencyAmount); ?>">
        <input type="hidden" name="accountingamount" value="<?php echo h($accountingCurrencyAmount); ?>">
    </form>

    <script>
        // Auto-submit
        document.forms.f1.submit();
    </script>

    <noscript>
        <p>JavaScript est désactivé. Cliquez pour continuer :</p>
        <button onclick="document.forms.f1.submit()">Continuer</button>
    </noscript>
</body>
</html>
